from median.models import Config, ListeModel, Magasin, ListeItemModel, Stock
from median.constant import TypeListe, TypeServiListe, PatientGlobal, EtatListe
from common.exception import RiedlDispatcherException
from peewee import DoesNotExist, fn
from common.util import logger, get_counter
from datetime import datetime


def get_url_dispatcher(r_poste: str) -> str:
    """
    Retrieve the URL of the dispatcher

    :param r_poste: Name of the poste relate to the equipement
    :type  r_poste: str
    :return: URL of the dispatcher
    :rtype: str
    """
    url = ""
    try:
        cfg = Config.get(poste=r_poste, cle="cfg", propriete="k_rdl_dispatcher_url")
        url = cfg.value
        if url.endswith("/"):
            url = url[:-1]
    except DoesNotExist:
        logger.error("dispatcher URL for rield [%s] does not exists" % (r_poste,))
        raise RiedlDispatcherException("dispatcher URL for rield %s does not exists" % (r_poste,))

    return url


def add_reference_inventory(poste: str, reference: str, gtin: str, username: str, liste_date: str) -> tuple[str, str]:
    """Add an item to an inventory list

    :param poste: Equipement to made the inventory
    :param reference: product reference to inventory
    :param gtin: Barcode for this reference
    :param username: User request the inventory
    """
    try:
        mag = Magasin.get(type_mag=poste)

        liste = "%s %s" % (mag.mag, liste_date)
        cpt = 0
        try:
            lst = ListeModel.get(mode=TypeListe.Inventory.value, liste=liste)
            cpt = lst.nb_item
        except DoesNotExist:
            lst = ListeModel()
            lst.mode = TypeListe.Inventory.value
            lst.liste = liste
            lst.etat = "V"
            lst.service = "TRAN"
            lst.type_servi = TypeServiListe.Inventory.value
            lst.id_servi = 8
            lst.fusion = "INVENTAIRE"
            lst.nb_item = 0
            lst.num_ipp = PatientGlobal.Ipp.value
            lst.num_sej = PatientGlobal.Sejour.value
            lst.zone_deb = "RIEDL"
            lst.zone_fin = poste
            lst.date_reception = datetime.now()
            lst.save()

        stk = Stock.select(
            fn.COUNT(Stock.pk).alias("boites"), fn.IFNULL(fn.SUM(Stock.quantite), 0).alias("total")
        ).where(Stock.magasin == mag.mag, Stock.reference == reference, Stock.cip == gtin)

        # TODO: add riedl counter to send the request
        try:
            itm = ListeItemModel.get(mode=TypeListe.Inventory.value, liste=liste, contenant=gtin)
            if itm.etat == EtatListe.Solde.value:
                itm.etat = EtatListe.Vierge.value
                itm.qte_dem = stk.qte_total
            itm.dtprise = datetime.now()
            itm.save()
            logger.info("item found %s" % itm.item)
        except DoesNotExist:
            cpt += 1
            logger.info("item new")
            itm = ListeItemModel()
            itm.mode = lst.mode
            itm.liste = lst.liste
            itm.item = "%06d" % cpt
            itm.dest = lst.service
            itm.magasin = mag.mag
            itm.qte_dem = stk[0].total
            itm.qte_serv = 0
            itm.quantite_disp = stk[0].boites
            itm.num_ipp = lst.num_ipp
            itm.num_sej = lst.num_sej
            itm.reference = reference
            itm.user = username
            itm.lot = ""
            itm.tperemp = ""
            itm.code_liv = gtin
            itm.contenant = gtin
            itm.type_servi = lst.type_servi
            itm.id_chargement = get_counter("RIEDL_INVENTORY")

            itm.save()

        lst.nb_item = cpt
        lst.save()

    except Exception as ex:
        print("oups %s" % str(ex))

    return lst
