from jinja2 import Environment, BaseLoader
import datetime
import time
import logging
import json

logger = logging.getLogger("median.print")


class RenderException(BaseException):
    def __init__(self, message):
        logger.error(f"RenderException: {message}")
        self.message = message


def _date_filter(value, format="%Y-%m-%d"):
    """Format the date to the specified format"""
    if isinstance(value, datetime.datetime):
        return value.strftime(format)
    elif isinstance(value, datetime.date):
        return value.strftime(format)
    return str(value)


def _time_filter(value, format="%H:%M:%S"):
    """Format the time to the specified format"""
    if isinstance(value, time.struct_time):
        return time.strftime(format, value)
    if isinstance(value, datetime.datetime):
        return value.strftime(format)

    return str(value)


def _float_filter(value, decimal=2):
    """Format the float to the specified number of decimals"""
    if isinstance(value, float):
        return f"{value:.{decimal}f}"
    elif isinstance(value, int):
        value = float(value)
        return f"{value:.{decimal}f}"
    elif isinstance(value, str):
        value = float(value)
        return f"{value:.{decimal}f}"

    return str(value)


def _truncate_filter(value, max_chars=0, trailing_dots=True):
    """Truncate the string to a maximum amount of characters"""
    if value is None:
        raise RenderException(f"Truncate filter failed, value is none. Hint : max_chars = {max_chars}")

    value = str(value)

    if max_chars == 0 or len(value) <= max_chars:
        return str(value)

    truncated_value = value[:max_chars].strip()

    if trailing_dots:
        return truncated_value.ljust(len(truncated_value) + 3, ".")
    else:
        return truncated_value


def _pad_filter(value, length=0, align="left", paddingChar=" "):
    """Pad the string with spaces (or given paddingChar) to reach exactly n characters.
    If value is longer than length, raise exception."""

    if value is None:
        raise RenderException(f"Pad filter failed, value is none. Hint : length = {length}")

    value = str(value)

    if len(value) > length:
        raise RenderException(f"Data loss in pad_filter: value '{value}' exceeds max length {length}")

    if align.lower() == "right":
        return value.rjust(length, paddingChar)
    elif align.lower() == "center":
        return value.center(length, paddingChar)
    else:  # Default: left align
        return value.ljust(length, paddingChar)


def _gtin_filter(value, max_chars=14):
    """Format a GTIN code to GS1 specifications"""
    if value is None:
        raise RenderException(f"GTIN filter failed, value is none. Hint : max_chars = {max_chars}")

    if len(str(value)) > max_chars:
        # Invalid value received, send only zeros
        return "".zfill(max_chars)

    return f"{value:>0{max_chars}}"  # leading zeros


def render_template(template_string: str, data_dict: dict, db_dict=None):
    """
    Renders a template string using Jinja2 templating engine.
    The template syntax follows Jinja2 conventions using double curly braces for variables: {{ variable }}.

    Filters can be applied to variables using the pipe symbol:
    {{ price|float_format(2) }} - formats a number as float with 2 decimal places
    {{ date|date_format('%d/%m/%Y') }} - formats a date with the specified format

    Args:
        template_string (str): The template string to render.
        data_dict (dict): Dictionary with dynamic data for rendering. If empty, preview mode is activated.
        db_dict (dict, optional): Dictionary containing template data from database (used for preview and key fallback).
                                 Keys from this dictionary will be used only if they don't exist in data_dict.
    Returns:
        str: The rendered template as a string.
    """
    try:
        if isinstance(data_dict, str):
            data_dict: dict = json.loads(data_dict)

        if db_dict is None:
            db_dict = {}
        elif isinstance(db_dict, str):
            db_dict: dict = json.loads(db_dict)

        preview_mode = not bool(data_dict)

        # In preview mode, only use db_dict
        # In regular mode, use data_dict and only include keys from db_dict that don't exist in data_dict
        if preview_mode:
            render_data = db_dict
        else:
            render_data = {}
            # Add keys from db_dict only if they don't exist in data_dict
            for key, value in db_dict.items():
                if key not in data_dict:
                    render_data[key] = value
            render_data.update(data_dict)

        env = Environment(loader=BaseLoader())
        env.filters["date_format"] = _date_filter
        env.filters["time_format"] = _time_filter
        env.filters["float_format"] = _float_filter
        env.filters["truncate_format"] = _truncate_filter
        env.filters["gtin_format"] = _gtin_filter
        env.filters["string_pad"] = _pad_filter

        template = env.from_string(template_string)
        output = template.render(render_data)

        return output
    except json.JSONDecodeError as e:
        raise RenderException(f"Error parsing JSON data: {str(e)}")
    except ValueError as e:
        raise RenderException(f"Error converting value: {str(e)}")
    except Exception as e:
        logger.error(type(e))
        raise RenderException(f"Unknown error: {str(e)}")
