import json
import logging
import html
from datetime import datetime

from flask import Blueprint, request, session
from flask_jwt_extended import jwt_required
from median import views as v
from median.base import BaseViewException
from median.constant import EcoType, TypeServiListe, DpmType, HistoryType, MEDIANWEB_POSTE, EtatListe
from median.models import (
    Config,
    ListeItemModel,
    ListeModel,
    Magasin,
    Product,
    Service,
    WorkItemModel,
    ListeErrorModel,
    Stock,
    Historique,
    Ucd,
    Gtin,
)
from median.views import RawConfig, SeuilView
from median.utils import get_counter
from peewee import DoesNotExist, JOIN
from ressources.equipments.externe import printLabels, reprintLabel
from median.database import mysql_db

from common.models import WebLogActions
from common.status import (
    HTTP_200_OK,
    HTTP_400_BAD_REQUEST,
    HTTP_404_NOT_FOUND,
    HTTP_500_INTERNAL_SERVER_ERROR,
)
from common.util import mustHaveRights

acced_replenish_blueprint = Blueprint("acced_replenish", __name__)

logger = logging.getLogger("median")

RESSOURCE_NAME = "WEB_REAPPRO"


@acced_replenish_blueprint.route("", methods=["POST"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def get():
    logger.info("Récupérer les listes de réappro...")

    data = json.loads(request.data)
    eco_types = (
        [EcoType.Externe.value]
        if data.get("reapproType") == "external"
        else [EcoType.Cueillette.value, EcoType.Coupe.value]
    )

    lis = (
        ListeModel.select(
            ListeModel.pk,
            ListeModel.liste,
            ListeModel.zone_fin,
            ListeModel.date_creation,
            Magasin.mag,
            ListeModel.etat,
            ListeModel.with_note,
            ListeErrorModel.message,
        )
        .where(ListeModel.mode == "E", ListeModel.zone_deb == "")
        .join(Magasin, on=ListeModel.zone_fin == Magasin.type_mag)
        .where(Magasin.eco_type << eco_types)
        .switch(ListeModel)
        .join(ListeErrorModel, JOIN.LEFT_OUTER, on=(ListeErrorModel.liste == ListeModel.liste))
        .order_by(ListeModel.zone_fin, ListeModel.pk)
    )
    logger.info("Récupérer les listes de réappro... %s" % lis.count())

    # Note: The memo at this date will come from a quasar WYSIWYG editor, thus it will be html. This is
    # encoded on save, to avoid XSS attacks.

    return [
        {
            "pk": li.pk,
            "label": li.liste,
            "mag": li.zone_fin,
            "date_creation": str(li.date_creation),
            "code_mag": li.magasin.mag,
            "etat": li.etat,
            "memo": html.unescape(li.listeerrormodel.message) if li.with_note > 0 else "",
        }
        for li in lis
    ], HTTP_200_OK


@acced_replenish_blueprint.route("<string:pk_liste>", methods=["POST"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def get_detail(pk_liste):
    # nb_item = 0
    try:
        args = json.loads(request.data) if request.data else {}
        v_limit = args["length"]
        v_offset = args["start"]
        v_filter_by_magasin = args["filterByMagasin"]
        v_search = args["search"].strip().upper() if args["search"] is not None else ""
    except KeyError as err:
        return {"message": "replenish.items.get.error", "args": str(err)}, HTTP_400_BAD_REQUEST
    except Exception as err:
        print(err)

    logger.info("Récupérer les items de listes de réappro, pk de la liste : '%s'" % pk_liste)

    try:
        pk_liste = int(pk_liste)
        # on récupère le libellé de la liste
        # c'est nul, mais c'est comme ça qu'on retrouve les items pour l'instant
        _liste = ListeModel.get_or_none((ListeModel.pk == pk_liste) & (ListeModel.mode == "E"))

        if not _liste:
            logger.error("No list for id %s" % pk_liste)
            return {"data": []}, HTTP_200_OK

        # nb_item = _liste[0].nb_item

        ms = [m for m in v_filter_by_magasin.split(",") if m]

        _items = (
            ListeItemModel.select(
                ListeItemModel.pk,
                ListeItemModel.etat,
                ListeItemModel.reference,
                ListeItemModel.fraction,
                Product.designation,
                Product.pk,
                ListeItemModel.qte_dem,
                ListeItemModel.qte_serv,
            )
            .join(Product, on=(Product.reference == ListeItemModel.reference))
            .where(
                (ListeItemModel.liste == _liste.liste)
                & (Product.designation.contains(v_search) if v_search else True)
                & (ListeItemModel.etat << ms if len(ms) > 0 else True)
            )
        )
        nb = _items.count()
        _items = _items.limit(v_limit).offset(v_offset).order_by(ListeItemModel.item)

        return {
            "recordsTotal": nb,
            "type_servi": _liste.type_servi,
            "data": [
                {
                    "pk": i.pk,
                    "etat": _get_etat_human_label(i.etat),
                    "reference": i.reference,
                    "reference_pk": i.product.pk,
                    "fraction": i.fraction,
                    "designation": i.product.designation,
                    "qte_demandee": i.qte_dem,
                    "qte_servie": i.qte_serv,
                }
                for i in _items
            ],
        }, HTTP_200_OK

    except Exception as error:
        logger.error("Get reappro items Datatables raised an exception: ", error.args)
        return {"messages": error.args}, HTTP_200_OK


@acced_replenish_blueprint.route("", methods=["DELETE"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canEdit=True)
def delete():
    args = request.args
    _pk = args["pk"]

    try:
        # get list label (needed to fetch items)
        liste: ListeModel = ListeModel.get_or_none(ListeModel.pk == _pk)

        if not liste:
            logger.error("Liste de réappro non trouvée: %s" % _pk)
            return {"message": "Liste de réappro non trouvée!"}, HTTP_404_NOT_FOUND

        liste.delete_instance()

        logger.info("Suppression réussie d'une liste de réappro: %s" % _pk)
        log_replenish(session["username"], "delete", f"Deleted replenish list : {_pk}")
        return {"message": "Success", "id": None}

    except Exception as error:
        logger.error(error.args)
        return {"message": error.args}, HTTP_400_BAD_REQUEST


@acced_replenish_blueprint.route("calcul_reappro", methods=["POST"])
@jwt_required()
def replenishment_calculation():
    logger.info("Calcul d'une liste d'entrée de reappro")

    args = json.loads(request.data)
    _mag_long = args["mag"]

    try:
        _m = Magasin.select(Magasin.mag).where(Magasin.type_mag == _mag_long)

        if len(_m) == 0:
            logger.error("Magasin non trouvé: " + _mag_long)
            return {"message": "Magasin non trouvé: " + _mag_long}, HTTP_400_BAD_REQUEST

        res = v.SeuilView().reappro_calcul(type_magasin=_mag_long, ignore_min_threshold=True)

    except Exception as error:
        logger.error(error.args)
        return {"message": error.args}, HTTP_400_BAD_REQUEST

    logger.info("Génération automatique d'une liste d'entrée... REUSSI")

    return res


@acced_replenish_blueprint.route("reappro", methods=["GET"])
@jwt_required()
def get_counter_replenishment():
    # n = Config.select(Config.value).where(Config.cle == 'CPT_LISTE_E' & Config.poste == 'TOUS')
    try:
        n = Config.get(poste="TOUS", cle="CPT_LISTE_S")
    except DoesNotExist as error:
        return {"message": error.args}, 400

    c = str(int(n.value) + 1)

    Config.update({Config.value: c}).where((Config.poste == "TOUS") & (Config.cle == "CPT_LISTE_S")).execute()

    return c.zfill(5)


@acced_replenish_blueprint.route("reappro", methods=["POST"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canEdit=True)
def create_replenishment():
    res = json.loads(request.data)

    liste = res["liste"]
    type_magasin = res["type_magasin"]

    # GPAO and global gpao send : if annule&remplace, the process should make all lists and items first
    # And when requested, a global GPAO generation should happen (k_annule_remplace == 0, gpao = false)
    config = RawConfig().read("k_annule_remplace")
    if config is None:
        gpao = True
    else:
        try:
            # parse the value to get a bool. '1', 1 and True are valid
            configValue = bool(int(config.value))
            gpao = configValue is False  # annule&remplace = 0, so gpao can be made (true)
        except (TypeError, ValueError):
            gpao = True
        except Exception as e:
            logger.error(str(e))
            return {"message": e.message, "error": str(e)}, HTTP_400_BAD_REQUEST

    username = session["username"]

    try:
        service = _transfert_service()
        mag = _check_type_magasin(type_magasin)
    except Exception as e:
        logger.error(str(e))
        return {"message": e.message, "error": str(e)}, HTTP_400_BAD_REQUEST

    # Création de la liste d'entrée
    if len(res["lines"]) > 0:
        list = ListeModel.create(
            liste=liste,
            date_creation=datetime.now(),
            etat="V",
            mode="E",
            fusion="REASSORT",
            service=service,
            date_modification=datetime.now(),
            zone_fin=type_magasin,
            type_servi=TypeServiListe.GlobaleBoite.value,
            id_servi=2,
            nb_item=len(res["lines"]),
        )

        # Création des items
        for r in res["lines"]:
            result = SeuilView().reappro_item(
                liste,
                r["item"],
                r["ref"],
                r["frac"],
                r["qte"],
                type_magasin,
                service,
                mag.id_zone,
                mag.id_robot,
                gpao,
                username,
            )
            if result is False:
                logger.error("Replenishment Create : Failed to create item, List cancelled")
                list.delete_instance()
                return {"pk": None}, HTTP_500_INTERNAL_SERVER_ERROR

        return {"pk": list.pk}, HTTP_200_OK
    else:
        # Création f_gpao
        if gpao:
            SeuilView().creer_gpao(mag.id_zone)

    return "ok"


@acced_replenish_blueprint.route("print_labels", methods=["PUT"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canEdit=True)
def external_printLabels():
    return printLabels()


@acced_replenish_blueprint.route("print_label/<string:serial>/<string:printer_pk>", methods=["GET"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canEdit=True)
def external_printLabel(serial, printer_pk):
    try:
        return reprintLabel(serial, printer_pk)
    except Exception as e:
        logger.error(str(e))
        return {}, HTTP_500_INTERNAL_SERVER_ERROR


def _transfert_service():
    """Retrieve the code of the service to transfert stock"""
    serviceConfig = RawConfig().read(param="k_ua_pui").value
    service = Service.get_or_none(Service.code == serviceConfig)

    if service is None:
        errmsg = "k_ua_pui not or badly configured"
        logger.error(errmsg)
        raise BaseViewException("Le magasin n'existe pas dans la base")

    return service.code


def _check_type_magasin(type_magas):
    """Check if magasin exists

    :param type_magas: code magasin a rechercher
    :type  type_magas: str
    :return: Magasin model
    :rtype: median.models.Magasin
    """
    try:
        m = Magasin.get(type_mag=type_magas)
    except DoesNotExist:
        logger.error("Le type magasin %s n'existe pas" % type_magas)
        raise BaseViewException("Le magasin n'existe pas dans la base")
    return m


def _get_etat_human_label(etat_short_label):
    options = {"V": "Vierge", "E": "Entamé", "S": "Soldé"}
    return options.get(etat_short_label, "Inconnu")


@acced_replenish_blueprint.route("workitems/<string:list_pk>/<string:item_pk>", methods=["GET"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def show_work_items(list_pk, item_pk):
    try:
        # x_urgent has been used to mark a workitem as "printed"
        # The ListeModel junction is unneccessary, but exists only as a safety mesure
        # to be sure that a serial is really linked to the active replenish list.

        work_items = (
            ListeModel.select(WorkItemModel.serial, WorkItemModel.urgent)
            .join(ListeItemModel, JOIN.INNER, on=(ListeModel.liste == ListeItemModel.liste))
            .join(WorkItemModel, JOIN.INNER, on=(ListeItemModel.pk == WorkItemModel.pk_item))
            .where((WorkItemModel.pk_item == item_pk) & (ListeModel.pk == list_pk))
        )

        serials = [
            {"serial": list.listeitemmodel.workitemmodel.serial, "printState": list.listeitemmodel.workitemmodel.urgent}
            for list in work_items
        ]
        return {"serials": serials}, HTTP_200_OK
    except Exception as error:
        logger.error(f"Error retrieving work items for item {item_pk}: {error}")
        return {"message": str(error)}, HTTP_400_BAD_REQUEST


@acced_replenish_blueprint.route("track/create", methods=["POST"])
@jwt_required()
@mustHaveRights(RESSOURCE_NAME, canView=True)
def create_track_base_stock():
    """With a GTIN and a DPM index, find the product and UCD, create a stock line and a histo line"""
    # TODO: We cast and we select one one line in ref_ucd or ucd_cip, because the database is full of crap
    # If able, consider fetching precisly the needed data instead of guessing.

    data = request.json

    try:
        index = data.get("index")
        product_data = data.get("product")
        box_data = data.get("box")
        liste_data = data.get("liste")
        item_data = data.get("item")

        batch = product_data["lot"]
        try:
            perem = datetime.strptime(product_data["perem"], "%y%m%d").date()
        except ValueError:
            perem = datetime.strptime(product_data["perem"], "%d/%m/%Y").date()
        productSerial = product_data.get("serial", "")  # Product serial is not always given

        boxCapa = box_data["capacity"]
        boxSerial = box_data["boxSerial"]

        liste_pk = liste_data["pk"]
        item_pk = item_data["pk"]

        with_delete = data.get("with_delete", False)
    except KeyError as err:
        return {"message": "replenish.track.create.error", "args": str(err)}, HTTP_400_BAD_REQUEST

    # Find the UCD
    try:
        casted_gtin = str(int(data["product"]["gtin"]))

        gtin: Gtin = Gtin.get(
            (Gtin.cip.cast("SIGNED").contains(casted_gtin))
            & (Gtin.dossier == index)
            & (Gtin.type_dpm == DpmType.Track.value)
        )

    except DoesNotExist:
        return {"message": "No trackable product for GTIN %s" % data["product"]["gtin"]}, HTTP_404_NOT_FOUND

    # Find the ref
    try:
        ucd: Ucd = Ucd.get(Ucd.ucd == gtin.ucd)
        product_data: Product = Product.get(Product.reference == ucd.reference)
    except DoesNotExist:
        return {"message": "No product for GTIN %s" % data["product"]["gtin"]}, HTTP_404_NOT_FOUND

    # Check if this box is already used in the stock
    boxStock: Stock = Stock.get_or_none(Stock.contenant == boxSerial)
    address = "K01"

    if with_delete:
        # User has been asked if we need to delete the existing stock, and responded positively
        log_replenish(
            session["username"],
            "track_delete_existing",
            f"User requested deletion of existing stock for box {boxSerial}",
        )
        logger.info(f"Delete existing stock for box {boxSerial}")
        boxStock.delete()
        boxStock = None

    if not boxStock:
        with mysql_db.atomic() as transaction:
            try:
                nowDate = datetime.now()

                liste: ListeModel = ListeModel.get(liste_pk)
                item: ListeItemModel = ListeItemModel.get(item_pk)

                # Create stock
                newStock: Stock = Stock()
                newStock.adresse = address
                newStock.reference = product_data.reference
                newStock.quantite = 0
                newStock.lot = batch
                newStock.date_peremption = perem
                newStock.date_entree = nowDate
                newStock.contenant = boxSerial
                newStock.magasin = address
                newStock.ucd = gtin.ucd
                newStock.cip = gtin.cip
                newStock.capa = boxCapa
                newStock.save()

                # Create histo
                trackCounter = "TRACK_" + str(get_counter("ID_TRACK", force_create=True)).zfill(12)
                newHisto: Historique = Historique()
                newHisto.chrono = nowDate
                newHisto.contenant = boxSerial
                newHisto.reference = product_data.reference
                newHisto.adresse = address
                newHisto.quantite_mouvement = 0
                newHisto.quantite_totale = 0
                newHisto.info = "Track - from " + liste.liste
                newHisto.liste = trackCounter
                newHisto.service = liste.service
                newHisto.type_mouvement = HistoryType.Coupe.value
                newHisto.lot = batch
                newHisto.utilisateur = session["username"]
                newHisto.date_peremption = perem
                newHisto.item = item.item
                # newHisto.ipp = liste.num_ipp
                # newHisto.sejour = liste.num_sej   # TODO: Check why it exists in the list
                newHisto.poste = MEDIANWEB_POSTE
                newHisto.pk_liste = liste.pk
                newHisto.ucd = gtin.ucd
                newHisto.pk_item = item.pk
                newHisto.item_wms = item.item_wms
                newHisto.serial = productSerial
                newHisto.save()

                # Update the replenishment list with the max possible quantity of a box
                item.qte_serv = int(item.qte_serv) + int(boxCapa)
                if item.qte_serv > item.qte_dem:
                    item.etat = EtatListe.Solde.value
                else:
                    item.etat = EtatListe.EnCours.value
                item.save()

            except Exception as err:
                transaction.rollback()
                logger.error(f"Track stock creation transaction error: {str(err)}")
                return {"message": "An error has occured"}, HTTP_500_INTERNAL_SERVER_ERROR
    else:
        # Box already used, abort !
        if boxStock.adresse == address:
            # Box already configured for the AideTrack
            return {"message": f"Box {boxSerial} already configured for AideTrack", "in_use": True}, HTTP_200_OK
        else:
            # Box used somewhere in the stock (shelves or machine)
            return {"message": f"Box {boxSerial} already used in stock", "in_use": True}, HTTP_200_OK

    return {"message": "Success"}, HTTP_200_OK


def log_replenish(username: str, action: str, message: str):
    """
    Add new log for replenish (liste)

    :param username: User made the action to log
    :param action:
    :param message: message to log
    """
    logger.info("Replenish[%s](%s)): %s" % (action, username, message))
    wlog = WebLogActions()
    wlog.chrono = datetime.now()
    wlog.username = username
    wlog.equipement_type = ""
    wlog.action = action
    wlog.message = message
    wlog.save()
