import json

from flask import Blueprint, request
from flask_jwt_extended import jwt_required
from median.models import Magasin, MagasinService, Service
from peewee import DoesNotExist, JOIN

from common.status import HTTP_200_OK, HTTP_400_BAD_REQUEST

astus_ward_blueprint = Blueprint('astus_ward', __name__)


@astus_ward_blueprint.route('associated/<string:astus_pk>', methods=['GET'])
@jwt_required()
def get_associated_astus(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)

        wards = MagasinService \
            .select(MagasinService, Service) \
            .join(Service, on=(MagasinService.dest == Service.code)) \
            .where((MagasinService.mag == mag.mag) & (Service.type_dest == "SERVICE"))

        list_ward = [{
                'pk': ward.service.pk,
                'code': ward.service.code,
                'label': ward.service.libelle
            } for ward in wards]

        return {
            'data': list_ward
        }, HTTP_200_OK

    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, HTTP_400_BAD_REQUEST


@astus_ward_blueprint.route('<string:astus_pk>', methods=['GET'])
@jwt_required()
def get_astus(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)

        wards = Service \
            .select(MagasinService, Service) \
            .join(MagasinService, JOIN.LEFT_OUTER,
                  on=((MagasinService.dest == Service.code) & (MagasinService.mag == mag.mag))) \
            .where((MagasinService.pk.is_null()) & (Service.type_dest == "SERVICE"))

        list_ward = [{
                'pk': ward.pk,
                'code': ward.code,
                'label': ward.libelle
            } for ward in wards]

        return {
            'data': list_ward
        }, HTTP_200_OK

    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, HTTP_400_BAD_REQUEST


@astus_ward_blueprint.route('<string:astus_pk>', methods=['POST'])
@jwt_required()
def add(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)
        data = json.loads(request.data)
        data_source = [
            {
                'mag': mag.mag,
                'dest': ward['code']
            } for ward in data
        ]
        MagasinService.insert_many(data_source).execute()
        return {}, HTTP_200_OK

    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, HTTP_400_BAD_REQUEST


@astus_ward_blueprint.route('<string:astus_pk>', methods=['DELETE'])
@jwt_required()
def delete(astus_pk):
    try:
        mag = Magasin.get(pk=astus_pk)
        data = json.loads(request.data)
        data_source = [ward['code'] for ward in data]

        MagasinService\
            .delete()\
            .where(
                (MagasinService.mag == mag.mag) &
                MagasinService.dest.in_(data_source))\
            .execute()
        return {}, HTTP_200_OK

    except DoesNotExist:
        return {'message': 'Magasin does not exist'}, HTTP_400_BAD_REQUEST
