import logging
import json

from common.util import get_counter
from flask import Blueprint, request
from flask_jwt_extended import jwt_required
from median.models import Adresse, CodeBlocage, Stock, Product
from peewee import JOIN, fn
from common.status import HTTP_200_OK

address_blueprint = Blueprint('address', __name__)

logger = logging.getLogger('median')


@address_blueprint.route('<string:adr>', methods=['POST'])
@jwt_required()
def get_container_counter(adr):
    logger.info('Retrieve a new container code for adress: "%s"' % adr)
    new_counter = get_counter('CONTENANT_PASS')
    logger.info('New container code for address "%s": "%09d"' % (adr, new_counter))
    return '%09d' % new_counter


@address_blueprint.route('all/<string:mag>', methods=['GET'])
@jwt_required()
def get_all(mag):
    try:
        adr = (Adresse.select(Adresse, CodeBlocage.libelle, Adresse.bloque_message,
                              Product.designation, fn.SUM(Stock.quantite).alias('qte_total'))
               .join(CodeBlocage, JOIN.LEFT_OUTER, on=CodeBlocage.valeur == Adresse.bloque)
               .join(Stock, JOIN.LEFT_OUTER, on=Stock.adresse == Adresse.adresse)
               .switch(Adresse)
               .join(Product, JOIN.LEFT_OUTER, on=Product.reference == Stock.reference)
               .where(Adresse.magasin == mag)
               .group_by(Adresse)
               .order_by(Adresse.adresse))

        logger.info('Lines : %s.' % len(adr))

    except Exception as error:
        logger.error('Exception levée en cherchant des adresses, '
                     'format: "%s", magasin: "%s". Message d\'erreur: "%s"' % (format, mag, error.args))
        return {'message': error.args}, 503

    if not len(adr):
        logger.warning('Adresses non trouvées, format: "%s", magasin: "%s"' % (format, mag))

    return ([{
        'adresse': a.adresse,
        'etat': a.etat,
        'contenant': a.contenant,
        'bloque': a.bloque,
        'designation': a.product.designation if hasattr(a, 'product') else '',
        'lock_custom_msg': a.bloque_message,
        'lock_msg': a.codeblocage.libelle if hasattr(a, 'codeblocage') else '',
        'pk': a.pk
    } for a in adr])


@address_blueprint.route("/selectbox", methods=["GET", "POST"])
@jwt_required()
def get_all_for_ref_field_selectbox():
    # This returns an array of options for a reference field (see freefields in config_blueprint)
    # Due to its dynamic nature, it HAS to return an array, named "data"
    # The params HAVE to be the name of an attribute of the peewee Adresse model (ex: eco_type, not x_eco_type)

    params = None
    if request.method == "POST":
        if request.data:
            params = json.loads(request.data)

    query = Adresse.select(Adresse.adresse).distinct()

    # Add where conditions based on params
    if params:
        for key, value in params.items():
            if hasattr(Adresse, key):
                query = query.where(getattr(Adresse, key) == value)
            else:
                logger.error(f'Address selectbox api : unknown attribute {key}')

    addresses = query.order_by(+Adresse.adresse)
    return {"data": [address.adresse for address in addresses]}, HTTP_200_OK
