from flask import Blueprint
from flask_jwt_extended import jwt_required

from median.models import CatalogType, CatalogVersion, CatalogOption

from common.status import HTTP_200_OK

from ressources.equipments.equipment import generator_topo_catalog

catalog_blueprint = Blueprint('catalog', __name__)


@catalog_blueprint.route('types', methods=['GET'])
@jwt_required()
def get_types():
    types = CatalogType.select(CatalogType.pk, CatalogType.type).where(CatalogType.isDisable == 0)

    return {
        'list': [{'pk': item.pk,
                  'label': item.type} for item in types]
    }, HTTP_200_OK


@catalog_blueprint.route('all', methods=['POST'])
@jwt_required()
def get_all_version():
    types = (CatalogVersion
             .select(CatalogVersion.numVersion, CatalogType.type,
                     CatalogVersion.type_pk, CatalogVersion.subVersion,
                     CatalogVersion.pk, CatalogVersion.isDisable)
             .join(CatalogType, on=CatalogVersion.type_pk == CatalogType.pk)
             .order_by(CatalogType.type, CatalogVersion.numVersion, CatalogVersion.subVersion.cast('SIGNED'))
             )

    return {
        'list': [{'pk': item.pk,
                  'label': item.numVersion,
                  'subVersion': item.subVersion,
                  'isDisable': item.isDisable,
                  'type_pk': item.type_pk,
                  'type': item.catalogtype.type}
                 for item in types]
    }, HTTP_200_OK


@catalog_blueprint.route('versions', methods=['GET'])
@jwt_required()
def get_all_versions():
    versions = (CatalogVersion
                .select(CatalogVersion.pk, CatalogVersion.numVersion, CatalogType.type,
                        CatalogVersion.subVersion)
                .join(CatalogType, on=CatalogType.pk == CatalogVersion.type_pk)
                .where((CatalogVersion.isDisable == 0))
                .order_by(CatalogType.type, CatalogVersion.numVersion, CatalogVersion.subVersion.cast('SIGNED')))

    return {
        'list': [{'pk': item.pk,
                  'type': item.catalogtype.type,
                  'numversion': item.numVersion,
                  'subversion': item.subVersion} for item in versions]
    }, HTTP_200_OK


@catalog_blueprint.route('<string:type_pk>/versions', methods=['GET'])
@jwt_required()
def get_versions(type_pk):
    versions = (CatalogVersion
                .select(CatalogVersion.pk, CatalogVersion.numVersion, CatalogVersion.subVersion)
                .where((CatalogVersion.isDisable == 0) &
                       (CatalogVersion.type_pk == type_pk))
                .order_by(CatalogVersion.numVersion, CatalogVersion.subVersion.cast('SIGNED')))

    return {
        'list': [{'pk': item.pk,
                  'label': item.numVersion,
                  'subLabel': item.subVersion} for item in versions]
    }, HTTP_200_OK


@catalog_blueprint.route('<string:type_pk>/<string:version_pk>/options', methods=['GET'])
@jwt_required()
def get_options(type_pk, version_pk):
    options = (CatalogOption.select(CatalogOption.pk, CatalogOption.type, CatalogOption.value)
                            .join(CatalogVersion, on=CatalogVersion.pk == CatalogOption.version_pk)
                            .where((CatalogVersion.type_pk == type_pk) &
                                   (CatalogOption.version_pk == version_pk)))

    return {
        'list': [{'pk': item.pk,
                  'label': item.type,
                  'value': item.value} for item in options]
    }, HTTP_200_OK


@catalog_blueprint.route('<string:version_pk>/topo', methods=['GET'])
@jwt_required()
def get_topo(version_pk):
    res = generator_topo_catalog(catalog_version_pk=version_pk)

    return {'list': res}, HTTP_200_OK
