import json
import os
import shutil
import uuid
import zipfile
import logging
from io import BytesIO
from pathlib import Path
from zipfile import ZipFile

from PIL import Image
from common.status import (
    HTTP_200_OK,
    HTTP_403_FORBIDDEN,
    HTTP_500_INTERNAL_SERVER_ERROR,
    HTTP_400_BAD_REQUEST,
    HTTP_201_CREATED,
    HTTP_404_NOT_FOUND,
)
from common.status import HTTP_204_NO_CONTENT
from flask import Blueprint, request, send_file
from flask_jwt_extended import jwt_required, get_jwt_identity
from median.constant import EcoType, DPM_VENTOUSE, DPM_PINCE, DPM_TRACK, DpmType
from median.models import User, Dpm, Magasin, DpmCheck, DpmAction, DpmComment, Gtin, Product, Ucd
from median.views import RawConfig
from peewee import DoesNotExist, JOIN, fn
from ressources.blueprint.dpm.action_dto import ActionDTO, CommentDTO
from ressources.blueprint.dpm.dpm_service import analyze_dpms, all_dpms_request, is_image
from ressources.blueprint.dpm.dpm_service import upload_dpms
from datetime import datetime

logger = logging.getLogger("median.webserver.dpm")
dpm_blueprint = Blueprint("dpm", __name__)

img_folder = os.environ.get("IMAGE_FOLDER", ".")

ZIP_MIMETYPE = ("application/zip", "application/octet-stream", "application/x-zip-compressed", "multipart/x-zip")


@dpm_blueprint.route("equipmenttypes", methods=["GET"])
@jwt_required()
def get_equipment_types():
    types = (
        Magasin.select(Magasin.type_machine)
        .where(Magasin.eco_type << [EcoType.Coupe.value, EcoType.Cueillette.value, EcoType.AideTrack.value])
        .group_by(Magasin.type_machine)
    )

    dpm_all = DPM_PINCE + DPM_VENTOUSE + DPM_TRACK

    return {
        "list": [{"type_machine": item.type_machine} for item in types if item.type_machine in dpm_all]
    }, HTTP_200_OK


@dpm_blueprint.route("types", methods=["GET"])
@jwt_required()
def get_types():
    equipments = (
        Magasin.select(Magasin.type_machine, Magasin.eco_type)
        .where(Magasin.eco_type << [EcoType.Coupe.value, EcoType.Cueillette.value])
        .group_by(Magasin.type_machine)
    )

    return {
        "list": [
            {"code": item.type_machine, "eco_type": item.eco_type, "type_machine": item.type_machine}
            for item in equipments
        ]
    }, HTTP_200_OK


@dpm_blueprint.route("count", methods=["POST"])
@jwt_required()
def get_dpms_count():
    dpms = all_dpms_request()
    return {"data": dpms.count()}, HTTP_200_OK


@dpm_blueprint.route("all", methods=["POST"])
@jwt_required()
def get_dpms():
    data = json.loads(request.data)
    page_number = data.get("pageNumber", 0)
    page_size = data.get("pageSize", -1)

    dpms = all_dpms_request().limit(page_size).offset(page_number * page_size).objects()

    return {
        "list": [
            {
                "pk": item.pk,
                "closeDate": item.close,
                "not_checked": item.not_checked,
                "equipment": {"type": item.equipment_type},
                "product": {
                    "reference": item.reference,
                    "designation": item.designation,
                    "cip": item.cip,
                    "index": item.dossier,
                    "gtin": item.gtin_pk,
                },
                "data": {
                    "ucd": item.ucd,
                    "qt_blister": item.qt_blister,
                    "qt_boite": item.qt_boite,
                    "qt_pass": item.qt_pass,
                    "type_du": item.DU_boite_pass,
                },
                "creator": {"date": item.creation, "name": item.creator, "avatar": item.creator_avatar},
                "test": {
                    "qty_cut": item.qt_coupe,
                    "date_last_cut": item.date_der_coupe,
                    "folder_status": get_folder_status(item),
                },
                "validation": {"date": item.validation, "name": item.validator, "avatar": item.validator_avatar},
                "modification": {
                    "date": item.modification,
                    "name": item.modification_user,
                    "avatar": item.modification_avatar,
                },
                "close": {"date": item.close, "name": item.close_user, "avatar": item.close_avatar},
            }
            for item in dpms
        ]
    }, HTTP_200_OK


def get_folder_status(dpm):
    cfg = RawConfig("MEDIANWEB").read("k_eco_dir_fiche_pince")
    pince_folder = cfg.value if cfg is not None else None
    cfg = RawConfig("MEDIANWEB").read("k_eco_dir_fiche_ventouse")
    ventouse_folder = cfg.value if cfg is not None else None

    if dpm.equipment_type in DPM_VENTOUSE:
        destination_folder = ventouse_folder
    elif dpm.equipment_type in DPM_PINCE:
        destination_folder = pince_folder
    else:
        logger.warning(f"Unknown equipment type: {dpm.equipment_type}")
        return False

    if destination_folder is None:
        logger.error("Destination folder not configured")
        return False

    path_to_test = os.path.join(destination_folder, dpm.cip, dpm.dossier)

    return os.path.exists(path_to_test)


@dpm_blueprint.route("<string:dpm_pk>/image/<string:type_image>", methods=["GET"])
@jwt_required()
def get_dpm_image(dpm_pk, type_image):
    identity = get_jwt_identity()

    try:
        User.select(User.pk).where((User.pk == identity)).get()
    except DoesNotExist:
        return {}, HTTP_403_FORBIDDEN

    dmp_obj = Dpm.select(Dpm.zip_file, Dpm.cip, Dpm.dossier).where(Dpm.pk == dpm_pk).get()

    cfg = RawConfig("MEDIANWEB").read("k_eco_dir_dpm")
    folder = cfg.value if cfg is not None else None
    path = f"{folder}\\{dmp_obj.zip_file}"

    if os.path.exists(path):
        with ZipFile(path) as dpmzip:
            info = dpmzip.infolist()
            img = next(
                filter(
                    lambda s: (not s.is_dir()) and (f"{dmp_obj.cip}/{dmp_obj.dossier}/{type_image}" in s.filename), info
                ),
                None,
            )

            img_buffer = dpmzip.read(img.filename)
            try:
                buf = BytesIO()
                img_byte_io = BytesIO(img_buffer)
                Image.open(img_byte_io).convert("RGB").save(buf, format="webp")
                buf.seek(0)
                return send_file(
                    buf, as_attachment=True, download_name=f"imagef{uuid.uuid4()}.webp", mimetype="image/webp"
                )
            except IOError:
                return {"message": "dpm.not_image.error"}, HTTP_500_INTERNAL_SERVER_ERROR

    return {}, HTTP_500_INTERNAL_SERVER_ERROR


@dpm_blueprint.route("<string:dpm_pk>/zip", methods=["GET"])
@jwt_required()
def get_dpm_zip(dpm_pk):
    identity = get_jwt_identity()
    try:
        User.select(User.pk).where(
            (User.pk == identity) & (User.profil << ["TECH", "PHARMACIEN", "ECO-DEX", "DEENOVA"])
        ).get()
    except DoesNotExist:
        return {}, HTTP_403_FORBIDDEN

    dmp_obj = Dpm.select(Dpm.zip_file, Dpm.cip, Dpm.dossier).where(Dpm.pk == dpm_pk).get()
    cfg = RawConfig("MEDIANWEB").read("k_eco_dir_dpm")
    folder = cfg.value if cfg is not None else None
    path = f"{folder}\\{dmp_obj.zip_file}"

    if os.path.exists(path):
        with open(path, "rb") as fh:
            buf = BytesIO(fh.read())
            buf.seek(0)
            return send_file(buf, as_attachment=True, download_name=f"{uuid.uuid4()}.zip", mimetype="multipart/x-zip")

    return {}, HTTP_500_INTERNAL_SERVER_ERROR


@dpm_blueprint.route("<string:dpm_pk>/check", methods=["PUT"])
@jwt_required()
def check_dpm(dpm_pk):
    try:
        dpm = Dpm.select().where(Dpm.pk == dpm_pk).get()
        identity = get_jwt_identity()
        usr = User.get(User.pk == identity)

        # Save the validation and close the DPM
        dpm.validation_date = datetime.now()
        dpm.validator_pk = usr.pk
        dpm.closing_date = datetime.now()
        dpm.closing_user_pk = usr.pk

        creator = User.get(User.pk == dpm.creator_pk)
        modifier = User.get(User.pk == dpm.modifier_pk) if dpm.modifier_pk else None
        validator = User.get(User.pk == dpm.validator_pk)

        checkers = []
        comments = []
        checker_set = set()

        try:
            for check in dpm.checks:
                actions = DpmAction.select().where(DpmAction.dpm_check == check.pk)
                for action in actions:
                    checker: User = User.get(User.pk == action.user_pk)
                    checker_domain = checker.login.split("@")

                    # Create a hashable key for the checker
                    checker_key = (
                        checker.username,
                        checker.profil,
                        checker_domain[1] if len(checker_domain) > 1 else "not found",
                    )

                    if checker_key not in checker_set:
                        checker_set.add(checker_key)
                        checkers.append(
                            {
                                "name": checker.username,
                                "profil": checker.profil,
                                "domain": checker_domain[1] if len(checker_domain) > 1 else "not found",
                            }
                        )

                    for comment in action.comments:
                        comments.append(
                            {"user": checker.username, "action": comment.dpm_action, "comment": comment.comment}
                        )

        except Exception as e:
            logger.error(f"Error retrieving checkers and comments: {str(e)}")

        # Prepare and append a text file (markdown format) with validation info
        cfg = RawConfig("MEDIANWEB").read("k_eco_dir_dpm")
        if not cfg or not cfg.value:
            return {"message": "DPM folder not configured"}, HTTP_500_INTERNAL_SERVER_ERROR

        path = f"{cfg.value}\\{dpm.zip_file}"

        if not os.path.exists(path):
            logger.error(f"ZIP File not found: {path}")
            return {"message": "DPM zip file not found"}, HTTP_404_NOT_FOUND

        try:
            with ZipFile(path, "a") as dpmzip:
                timestamp = format_date_for_md(datetime.now())
                md_path = f"{dpm.cip}/{dpm.dossier}/validated_by_{usr.username}_{timestamp}.txt"

                try:
                    info = zipfile.ZipInfo(md_path, datetime.now().timetuple()[:6])
                    info.compress_type = zipfile.ZIP_DEFLATED

                    modifier_text = ""
                    if modifier:
                        modifier_text = "Last modified by: "
                        modifier_text += f"{modifier.username} at {format_date_for_md(dpm.modification_date)}\n"

                    with dpmzip.open(info, "w") as mdfile:
                        validation_text = (
                            f"## DPM Information\n"
                            f"- CIP: {dpm.cip}\n"
                            f"- Version: {dpm.dossier}\n"
                            f"- Machine type: {dpm.type_machine}\n"
                            f"\n"
                            f"## DPM Data\n"
                            f"- UCD: {dpm.ucd}\n"
                            f"- Qty Blister: {dpm.qt_blister}\n"
                            f"- Qty Box: {dpm.qt_boite}\n"
                            f"- Qty Pass Box: {dpm.qt_pass}\n"
                            f"- Unit Dose Type: {dpm.DU_boite_pass}\n"
                            f"\n"
                            f"## DPM Life cycle\n"
                            f"- Created by: {creator.username} at {format_date_for_md(dpm.creation)}\n"
                            f"- {modifier_text}"
                            f"- Validated by: {validator.username} at {format_date_for_md(dpm.validation_date)}\n"
                            f"\n"
                        )

                        # Append checkers and comments
                        validation_text += "## Checkers\n"
                        for checker in checkers:
                            validation_text += (
                                f"- {checker['name']} ({checker['profil']}) "
                                f"from {checker['domain'] if checker['domain'] else 'unknown domain'}\n"
                            )

                        validation_text += "\n"  # newline

                        if len(comments) > 0:
                            validation_text += "## Comments\n"
                            for comment in comments:
                                validation_text += (
                                    f"- On action n°{comment['action']} from {comment['user']}: {comment['comment']}\n"
                                )

                        mdfile.write(validation_text.encode("utf-8"))
                except Exception as e:
                    logger.error(f"Error writing to markdown file: {str(e)}")
                    return {"message": "Error writing validation file"}, HTTP_500_INTERNAL_SERVER_ERROR

        except Exception as e:
            logger.error(f"Error opening zip file: {str(e)}")
            return {"message": "Error accessing DPM zip file"}, HTTP_500_INTERNAL_SERVER_ERROR

        # Save the modified dpm record
        dpm.save()

        return {}, HTTP_204_NO_CONTENT

    except DoesNotExist as e:
        logger.error(f"DPM or user not found: {str(e)}")
        return {"message": "DPM or user not found"}, HTTP_404_NOT_FOUND
    except Exception as e:
        logger.error(f"Unexpected error in check_dpm: {str(e)}")
        return {"message": "Internal server error"}, HTTP_500_INTERNAL_SERVER_ERROR


def format_date_for_md(date):
    return date.strftime("%Y-%m-%d %H:%M:%S")


@dpm_blueprint.route("<string:dpm_pk>/checks", methods=["GET"])
@jwt_required()
def get_dpm_checks(dpm_pk):
    User_action = User.alias("userAction")
    User_comment = User.alias("userComment")

    checks = (
        DpmCheck.select(
            DpmCheck.profil.alias("profil"),
            Dpm.zip_file.alias("zip_file"),
            Dpm.cip.alias("cip"),
            Dpm.dossier.alias("index"),
            DpmCheck.num.alias("num"),
            DpmCheck.is_checked.alias("is_checked"),
            DpmCheck.edit_date.alias("edit_date"),
            User.username.alias("username"),
            User.avatar.alias("avatar"),
            DpmAction.num.alias("action_num"),
            DpmAction.is_checked.alias("action_checked"),
            DpmAction.pk.alias("action_pk"),
            DpmAction.edit_date.alias("action_edit"),
            User_action.avatar.alias("action_avatar"),
            User_action.username.alias("action_username"),
            DpmComment.pk.alias("comment_pk"),
            DpmComment.creation_date.alias("comment_creation"),
            DpmComment.comment.alias("comment_msg"),
            User_comment.avatar.alias("comment_avatar"),
            User_comment.username.alias("comment_username"),
        )
        .join(Dpm)
        .switch(DpmCheck)
        .join(DpmAction)
        .join(DpmComment, JOIN.LEFT_OUTER)
        .join(User, JOIN.LEFT_OUTER, on=User.pk == DpmCheck.user_pk)
        .join(User_action, JOIN.LEFT_OUTER, on=User_action.pk == DpmAction.user_pk)
        .join(User_comment, JOIN.LEFT_OUTER, on=User_comment.pk == DpmComment.user_pk)
        .where(Dpm.pk == dpm_pk)
        .order_by(DpmCheck.profil, DpmCheck.num, DpmAction.num, DpmComment.creation_date)
    ).objects()
    res = []
    links = None
    cfg = RawConfig("MEDIANWEB").read("k_eco_dir_dpm")
    folder = cfg.value if cfg is not None else None

    for item in checks:
        if links is None:
            links = []
            path = f"{folder}\\{item.zip_file}"

            if os.path.exists(path):
                with ZipFile(path) as dpmzip:
                    info = dpmzip.infolist()
                    for file in filter(lambda s: not s.is_dir(), info):
                        img_buffer = dpmzip.read(file.filename)

                        if is_image(img_buffer):
                            links.append(os.path.basename(file.filename).split(".")[0])

        profil = next(filter(lambda s: s["profil"] == item.profil, res), None)

        if profil is None:
            profil = {"profil": item.profil, "checks": []}
            res.append(profil)

        check = next(filter(lambda s: s["num"] == item.num, profil["checks"]), None)

        if check is None:
            check = {
                "isChecked": item.is_checked,
                "user": {
                    "name": item.username,
                    "avatar": item.avatar,
                },
                "edit": item.edit_date,
                "num": item.num,
                "actions": [],
            }
            profil["checks"].append(check)

        action = next(filter(lambda s: s["pk"] == item.action_pk, check["actions"]), None)

        if action is None:
            action = json.loads(
                ActionDTO(
                    pk=item.action_pk,
                    is_checked=item.action_checked,
                    num=item.action_num,
                    edit=item.action_edit,
                    avatar=item.action_avatar,
                    username=item.action_username,
                ).toJson()
            )
            check["actions"].append(action)

        if item.comment_pk is not None:
            action["comments"].append(
                json.loads(
                    CommentDTO(
                        pk=item.comment_pk,
                        creation=item.comment_creation,
                        comment=item.comment_msg,
                        avatar=item.comment_avatar,
                        username=item.comment_username,
                    ).toJson()
                )
            )

    return {"links": links, "list": res}, HTTP_200_OK


@dpm_blueprint.route("analyze", methods=["POST"])
@jwt_required()
def analyze():
    try:
        dpms = analyze_dpms(request)

        result_list = []
        for dpm in dpms:
            parsed = json.loads(dpm.toJson())
            result_list.append(parsed)

        return {"list": result_list}, HTTP_200_OK
    except Exception as e:
        return {"message": str(e), "trace": repr(e.args)}, HTTP_500_INTERNAL_SERVER_ERROR


@dpm_blueprint.route("import", methods=["POST"])
@jwt_required()
def upload():
    try:
        result = upload_dpms(request, get_jwt_identity())
        return {"list": result}, HTTP_201_CREATED
    except Exception as e:
        logger.error(f"MSR Upload : {e.args}")
        return {"message": "dpm.error.upload_failed"}, HTTP_500_INTERNAL_SERVER_ERROR


@dpm_blueprint.route("<string:dpm_action_pk>/comments", methods=["PUT"])
@jwt_required()
def add_comment(dpm_action_pk):
    data = json.loads(request.data)
    comment = data.get("value", None)

    user_id = get_jwt_identity()
    try:
        usr = User.select(User.profil, User.avatar, User.username).where((user_id == User.pk)).get()
    except DoesNotExist:
        return {"message": "dpm.check.error.forbidden"}, HTTP_403_FORBIDDEN

    action = DpmAction.select().where(DpmAction.pk == dpm_action_pk).get()

    dpm_comment = DpmComment()
    dpm_comment.comment = comment
    dpm_comment.dpm_action = action
    dpm_comment.creation_date = datetime.now()
    dpm_comment.user_pk = user_id
    dpm_comment.save()

    json_comment = CommentDTO(
        pk=dpm_comment.pk,
        creation=dpm_comment.creation_date,
        comment=dpm_comment.comment,
        avatar=usr.avatar,
        username=usr.username,
    )

    return {"data": json.loads(json_comment.toJson())}


@dpm_blueprint.route("<string:dpm_action_pk>", methods=["PUT"])
@jwt_required()
def update_check(dpm_action_pk):
    data = json.loads(request.data)
    value = data.get("value", None)

    check = DpmCheck.select(DpmCheck.profil).join(DpmAction).where(DpmAction.pk == dpm_action_pk).get()

    user_id = get_jwt_identity()
    try:
        allowed_profiles = [check.profil]
        if check.profil == "TECH":
            allowed_profiles.extend(["DEENOVA", "ECO-DEX"])

        usr = (
            User.select(User.profil, User.avatar, User.username)
            .where((user_id == User.pk) & (User.profil << allowed_profiles))
            .get()
        )
    except DoesNotExist:
        return {"message": "dpm.check.error.forbidden"}, HTTP_403_FORBIDDEN

    action = DpmAction.get(pk=dpm_action_pk)
    action.is_checked = value
    action.edit_date = datetime.now()
    action.user_pk = user_id
    action.save()

    json_action = ActionDTO(
        pk=action.pk,
        is_checked=action.is_checked,
        num=action.num,
        edit=action.edit_date,
        avatar=usr.avatar,
        username=usr.username,
    )

    return {"data": json.loads(json_action.toJson())}


@dpm_blueprint.route("sendtotest", methods=["POST"])
@jwt_required()
def send_to_test():
    try:
        data = json.loads(request.data)
        dpm_pks = data.get("dpm_pks", [])
    except Exception as ex:
        return {"message": "dpm.check.error.forbidden", "trace": "%s" % repr(ex.args)}, HTTP_403_FORBIDDEN

    try:
        user_id = get_jwt_identity()
        User.select(User.profil, User.avatar, User.username).where((user_id == User.pk)).get()
    except DoesNotExist:
        return {"message": "dpm.check.error.forbidden", "trace": "User doesn't exist"}, HTTP_403_FORBIDDEN

    # Retrieve all folders for all types of cutting files
    cfg = RawConfig("MEDIANWEB").read("k_eco_dir_dpm")
    source_folder = cfg.value if cfg is not None else None
    cfg = RawConfig("MEDIANWEB").read("k_eco_dir_fiche_pince")
    pince_folder = cfg.value if cfg is not None else None
    cfg = RawConfig("MEDIANWEB").read("k_eco_dir_fiche_ventouse")
    ventouse_folder = cfg.value if cfg is not None else None

    if not ventouse_folder and not pince_folder:
        return {
            "message": "dpm.check.error.folder_not_configured",
            "trace": "Folder not configured",
        }, HTTP_500_INTERNAL_SERVER_ERROR

    # Read all DPM's
    dpms = Dpm.select(
        Dpm.pk,
        Dpm.zip_file,
        Dpm.type_machine,
        Dpm.cip,
        Dpm.ucd,
        Dpm.dossier,
        Dpm.qt_blister,
        Dpm.qt_boite,
        Dpm.qt_pass,
        Dpm.DU_boite_pass,
    ).where(Dpm.pk << dpm_pks)

    count_success_dpm_sent = 0
    dpms_send_status = []

    dpm: Dpm
    for dpm in dpms:
        dpm_send_status = {"cip": dpm.cip, "status": False, "message": ""}
        dpms_send_status.append(dpm_send_status)

        # Check for the required values
        if (
            dpm.ucd == ""
            or dpm.qt_blister == 0
            or dpm.qt_boite == 0
            or dpm.qt_pass == 0
            or dpm.DU_boite_pass not in [0, 1]
        ):
            logger.warning(f"Missing values for DPM with pk: {dpm.pk}")
            dpm_send_status["message"] = "dpm.test.send.error.missing_values"
            continue

        # Check for valid UCD
        pro_ucd: Ucd = Ucd.get_or_none(Ucd.ucd == dpm.ucd)
        if pro_ucd is None:
            logger.warning(f"UCD unknown for DPM with pk: {dpm.pk}")
            dpm_send_status["message"] = "dpm.test.send.error.ucd_unknown"
            continue

        product: Product = Product.get_or_none(Product.reference == pro_ucd.reference)
        if product is None:
            logger.warning(f"Reference unknown for UCD: {pro_ucd.reference}")
            dpm_send_status["message"] = "dpm.test.send.error.pro_unknown"
            continue

        if dpm.type_machine in DPM_VENTOUSE:
            destination_folder = ventouse_folder
            type_dpm = DpmType.Ventouse.value
        elif dpm.type_machine in DPM_PINCE:
            destination_folder = pince_folder
            type_dpm = DpmType.Pince.value
        else:
            destination_folder = None
            type_dpm = DpmType.Pince.value

        if destination_folder:
            Path(f"{destination_folder}").mkdir(parents=True, exist_ok=True)

            if os.path.exists(f"{destination_folder}/{dpm.cip}/{dpm.dossier}"):
                shutil.rmtree(f"{destination_folder}/{dpm.cip}/{dpm.dossier}")

            if os.path.exists(f"{source_folder}/{dpm.zip_file}"):
                with zipfile.ZipFile(f"{source_folder}/{dpm.zip_file}", "r") as zip_ref:
                    zip_ref.extractall(f"{destination_folder}")
            else:
                logger.warning(f"Source files missing for DPM: {dpm.zip_file} in {source_folder}")
                dpm_send_status["message"] = "dpm.test.send.error.source_files_missing"
                continue

            # Add the DPM to the original table for testing
            try:
                gtin, _ = Gtin.get_or_create(
                    cip=dpm.cip,
                    dossier=dpm.dossier,
                    defaults={
                        "ucd": dpm.ucd,
                        "chrono": datetime.now(),
                        "qt_blister": dpm.qt_blister,
                        "qt_boite": dpm.qt_boite,
                        "qt_pass": dpm.qt_pass,
                        "DU_boite_pass": dpm.DU_boite_pass,
                        "last_user": "MEDIANWEB",
                        "type_dpm": type_dpm,
                    },
                )
                dpm.ucd_cip = gtin.pk
                logger.info(dpm.pk)
                dpm.save()
                dpm_send_status["status"] = True

                product.reap_mode = "T"  # TODO: Use median.constant
                product.save()

                count_success_dpm_sent += 1
            except Exception as e:
                logger.warning(f"Error for DPM with pk: {dpm.pk}: {e.args}")
                dpm_send_status["message"] = "dpm.test.send.error.generic_error"
                continue

    return {
        "data": {"qty_requested": len(dpms), "qty_success": count_success_dpm_sent, "data": dpms_send_status}
    }, HTTP_200_OK


@dpm_blueprint.route("<string:dpm_pk>/updatedata", methods=["PATCH"])
@jwt_required()
def update_data(dpm_pk):
    try:
        data = request.get_json()
        if "type" not in data or "value" not in data:
            return {"error": True, "message": "dpm.error.invalid_data_type"}, HTTP_400_BAD_REQUEST
        newValue = data["value"]
        valueType = data["type"]

        user_id = get_jwt_identity()

        dpm = (
            Dpm.select(Dpm, fn.COALESCE(Gtin.qt_coupe, 0).alias("qt_coupe"))
            .join(Gtin, JOIN.LEFT_OUTER, on=(Gtin.pk == Dpm.ucd_cip))
            .where(Dpm.pk == dpm_pk)
            .first()
        )

        if dpm.validator_pk or dpm.closing_user_pk or (hasattr(dpm, "gtin") and dpm.gtin and dpm.gtin.qt_coupe > 0):
            return {"message": "DPM Validated"}, HTTP_400_BAD_REQUEST

        if valueType == "ucd":
            ucd = Ucd.get_or_none(Ucd.ucd == newValue)
            if ucd:
                dpm.ucd = newValue
            else:
                return {"message": "UCD Unknown"}, HTTP_400_BAD_REQUEST
        elif valueType == "qt_blister":
            dpm.qt_blister = newValue
        elif valueType == "qt_boite":
            dpm.qt_boite = newValue
        elif valueType == "qt_pass":
            dpm.qt_pass = newValue
        elif valueType == "type_du":
            dpm.DU_boite_pass = newValue
        else:
            return {"message": "Invalid data type"}, HTTP_400_BAD_REQUEST

        dpm.modifier_pk = user_id
        dpm.modification_date = datetime.now()

        dpm.save()

        return {}, HTTP_204_NO_CONTENT

    except Exception as error:
        logger.error(error.args)
        return {"message": "dpm.update.error", "trace": repr(error.args)}, HTTP_500_INTERNAL_SERVER_ERROR
