import json
import logging
import math
from datetime import datetime

from common.status import (HTTP_200_OK, HTTP_500_INTERNAL_SERVER_ERROR, HTTP_404_NOT_FOUND, HTTP_400_BAD_REQUEST,
                           HTTP_405_METHOD_NOT_ALLOWED)
from dateutil.relativedelta import relativedelta
from flask import Blueprint, request, send_file, session
from flask_jwt_extended import jwt_required, get_jwt_identity
from median.constant import EtatAdresse
from median.models import Adresse, CodeBlocage, Stock, HistoriqueBlocage, User, Profil
from peewee import DoesNotExist, JOIN, fn
from ressources.astus.utils import generate_memory_excel_file
from ressources.blueprint.locations.locations_service import get_obj_locations, get_obj_locations_export, \
    get_all_locations, get_all_format

from common.status import HTTP_204_NO_CONTENT, HTTP_403_FORBIDDEN
from ressources.blueprint.locations.locations_service import stock_mvt

locations_blueprint = Blueprint('locations', __name__)

logger = logging.getLogger('median')


@locations_blueprint.route('formats', methods=['POST'])
@jwt_required()
def get_formats():
    try:
        args = json.loads(request.data)
        return {
            'formats': get_all_format(args)
        }, HTTP_200_OK
    except Exception as e:
        return {
            'formats': [],
            'error': e.args
        }, HTTP_500_INTERNAL_SERVER_ERROR


@locations_blueprint.route('', methods=['POST'])
@jwt_required()
def get_locations():
    try:
        args = json.loads(request.data)
        v_start = args['start']
        v_length = args['length']
        total_stocks_count = get_all_locations(args)
        paged_emplacements = total_stocks_count.limit(v_length).offset(v_start)

        listLoc = []

        for loc in paged_emplacements:
            listLoc.append(get_obj_locations(loc))

        return {
            'data': listLoc,
            'recordsTotal': total_stocks_count.count(),
        }

    except DoesNotExist:
        logger.error('Get reappro items Datatables raised a DoesNotExist exception')
        return {
            'data': [],
            'recordsTotal': 0,
        }
    except AttributeError:
        return {
            'data': [],
            'recordsTotal': 0,
            'recordsFiltered': 0,
            'error': 'filterByMagasin missing from the request'
        }
    except Exception as error:
        logger.error(('Datatables emplacements raised an exception: ', error.args))
        return {
            'data': [],
            'recordsTotal': 0,
            'error': error.args
        }


@locations_blueprint.route('export', methods=['PATCH'])
@jwt_required()
def export():
    data = json.loads(request.data)
    headers = data['headers']

    stocks = get_all_locations(data)

    memory_file = generate_memory_excel_file(headers=headers, items=stocks, transform_function=get_obj_locations_export)

    if memory_file is None:
        return {'message': 'generic.excel.export.error'}, HTTP_500_INTERNAL_SERVER_ERROR
    else:
        return send_file(memory_file, as_attachment=True, download_name="location.xlsx")


@locations_blueprint.route('comments', methods=['GET'])
@jwt_required()
def get_messages():
    try:
        req = CodeBlocage \
            .select(CodeBlocage.valeur, CodeBlocage.libelle, CodeBlocage.manuel) \
            .order_by(CodeBlocage.libelle)
        return {
            'data': [{
                'label': msg.libelle,
                'pk': msg.valeur,
                'manual': msg.manuel
            } for msg in req],
        }, HTTP_200_OK
    except Exception as e:
        return {
            'data': [],
            'error': e.args
        }, HTTP_500_INTERNAL_SERVER_ERROR


@locations_blueprint.route('', methods=['GET'])
@jwt_required()
def get():
    return {}, HTTP_405_METHOD_NOT_ALLOWED


@locations_blueprint.route('', methods=['PUT'])
@jwt_required()
def update_location():
    """
    Update an adress to block or release
    If address is multi product, we only update stock
    """
    try:
        args = json.loads(request.data)
        adr = args.get('adr', '')
        is_Buffer = args.get('isBuffer', False)
        bloque = args['bloque']
        etat = args.get('state', EtatAdresse.Libre.value)
        pk = args['pk']
        comment = args.get('comment', '')

    except Exception as err:
        return {
            'message': err.args,
        }, HTTP_500_INTERNAL_SERVER_ERROR

    if not adr:
        logger.warning('Adresse %s manquante!' % adr)
        return {'message': 'Adresse manquante'}, HTTP_404_NOT_FOUND

    if bloque is None:
        logger.warning('Blocking code not found for address %s!', adr)
        return {'message': 'Blocking code not found'}, HTTP_404_NOT_FOUND
    else:
        bloque = int(bloque)

    etat = bloque > 0 and EtatAdresse.Bloque.value or etat
    stocks = Stock.select(Stock.pk).where(Stock.adresse == adr).count()

    if stocks > 0 and etat in (EtatAdresse.Libre.value, 'U'):
        etat = EtatAdresse.Occupe.value

    addr_multi = False
    logger.info('Blocage/Déblocage adresse: "%s"' % (adr))
    try:
        if not is_Buffer:
            addr = Adresse.get(Adresse.adresse == adr)
            if addr.etat == EtatAdresse.Multiple.value:
                addr_multi = True

            if addr_multi:
                n = (Stock.update({Stock.bloque: bloque}).where(
                    Stock.pk == pk))
                n.execute()
            else:
                n = (Stock.update({Stock.bloque: bloque}).where(
                    Stock.adresse == adr))
                n.execute()

            if not addr_multi:
                n = (Adresse.update({
                    Adresse.etat: etat,
                    Adresse.bloque: bloque,
                    Adresse.bloque_message: comment
                }).where(Adresse.adresse == adr))
                n.execute()

            try:
                stock = (Stock.select().where(
                    Stock.adresse == adr)).get()
            except DoesNotExist:
                stock = None
        else:
            stock = ((Stock.select()
                      .where((Stock.adresse == adr) & (Stock.pk == pk)))
                     .get())
            stock.bloque = bloque
            stock.save()

        # Write an event on blocking history table (for lock and unlock)
        HistoriqueBlocage.create(
            adresse=adr,
            chrono=datetime.now(),
            bloque=bloque,
            comment=comment,
            datalu="Mweb, %s,%s" % (comment or '-', session['username']),
            reference=stock and stock.reference or '',
            date_peremption=stock and stock.date_peremption or '',
            lot=stock and stock.lot or '',
            fraction=stock and stock.fraction or '',
            ucd=stock and stock.ucd or '',
        )

    except DoesNotExist:
        logger.error('Address not exists: "%s"' % (adr))
        return {'message': 'Address not exists'}, HTTP_404_NOT_FOUND
    except Exception as error:
        logger.error(error.args)
        return {'message': error.args}, HTTP_400_BAD_REQUEST

    logger.info("""L'adresse a été modifié sans problème. Réf: "%s" """ % (adr))
    return {'result': 'OK', 'address': adr, 'pk': pk, 'state': etat}, HTTP_200_OK


@locations_blueprint.route('<string:stock_pk>', methods=['PUT'])
@jwt_required()
def update_quantity(stock_pk):
    data = json.loads(request.data)
    new_qty = data.get('quantity', None)

    identity = get_jwt_identity()
    try:
        usr = (User.select()
                   .join(Profil, on=Profil.profil == User.profil)
                   .where((User.pk == identity) &
                          (Profil.ressource == 'WEB_LOCATIONS') &
                          (Profil.edit == 1))).get()
    except DoesNotExist:
        return {}, HTTP_403_FORBIDDEN

    if new_qty is None or math.isnan(float(new_qty)):
        return {'alertMessage': 'location.quantity.error.empty'}, HTTP_500_INTERNAL_SERVER_ERROR

    stk = Stock.get(pk=stock_pk)
    if float(new_qty) - stk.quantite != 0:
        stock_mvt(stk=stk, diff=float(new_qty) - stk.quantite, usr=usr)
        stk.quantite = float(new_qty)
        stk.save()

    return {}, HTTP_204_NO_CONTENT


@locations_blueprint.route('<string:adr>/blocked', methods=['POST'])
@jwt_required()
def get_blocked_history(adr):
    data = json.loads(request.data)
    start = data['start']
    end = data['end']

    date_end = datetime.strptime(end, "%Y-%m-%d") + relativedelta(days=1)

    req = (HistoriqueBlocage.select(fn.DATE(HistoriqueBlocage.chrono).alias('date'),
                                    HistoriqueBlocage.bloque.alias('bloque'),
                                    CodeBlocage.libelle.alias('label'),
                                    fn.COUNT(HistoriqueBlocage.pk).alias('count'))
           .join(CodeBlocage, JOIN.LEFT_OUTER, on=CodeBlocage.valeur == HistoriqueBlocage.bloque)
           .where((HistoriqueBlocage.adresse == adr) &
                  (HistoriqueBlocage.chrono >= start) &
                  (HistoriqueBlocage.chrono < date_end))
           .group_by(fn.DATE(HistoriqueBlocage.chrono), HistoriqueBlocage.bloque, CodeBlocage.libelle)
           .order_by(fn.DATE(HistoriqueBlocage.chrono)))
    elts = []
    for item in req.objects():
        utc_item_date = datetime.utcfromtimestamp(datetime.combine(item.date, datetime.min.time()).timestamp())
        date = next(filter(lambda e: e['date'] == utc_item_date, elts), None)

        if date is None:
            date = {
                'date': utc_item_date,
                'blocks': []
            }
            elts.append(date)

        date['blocks'].append({
            'code': item.bloque,
            'id': item.bloque,
            'count': item.count,
            'label': item.label if item.label is not None else None
        })

    return {
        'list': elts
    }, HTTP_200_OK
