import logging
from datetime import datetime
from flask import Blueprint, request, session
from flask_jwt_extended import jwt_required
from median.database import mysql_db
from median.models import Magasin, MagasinGroupe
from peewee import DoesNotExist, IntegrityError
from common.models import WebLogActions
from common.status import (
    HTTP_200_OK, HTTP_204_NO_CONTENT, HTTP_500_INTERNAL_SERVER_ERROR,
    HTTP_409_CONFLICT, HTTP_404_NOT_FOUND)

maggroups_blueprint = Blueprint('maggroups', __name__)

logger = logging.getLogger('median.log')


@maggroups_blueprint.route('', methods=['GET'])
@jwt_required()
def get_all():
    try:
        query = MagasinGroupe.select().order_by(MagasinGroupe.code.asc())

        groups = [{
            'pk': i.pk,
            'code': i.code,
            'name': i.nom,
        } for i in query]

        return {'groups': groups, 'groupcount': query.count()}, HTTP_200_OK

    except Exception as error:
        logger.error('Get all groups raised an exception: %s', error.args)
        return {'message': 'maggroups.get_all.error'}, HTTP_500_INTERNAL_SERVER_ERROR


@maggroups_blueprint.route('<string:id>', methods=['GET'])
@jwt_required()
def get_group(id):
    try:
        group = MagasinGroupe.get(pk=id)

        code = group.code

        stores = Magasin.select(
            Magasin.pk, Magasin.mag, Magasin.type_machine, Magasin.libelle
        ).where(
            Magasin.groupe == code
        ).order_by(
            Magasin.pk.asc()
        )

        return {
            'pk': group.pk,
            'code': group.code,
            'name': group.nom,
            'stores': [{
                'pk': store.pk,
                'mag': store.mag,
                'type_machine': store.type_machine,
                'label': store.libelle
            } for store in stores]
        }

    except DoesNotExist:
        return {'message': 'maggroups.error.not_exist'}, HTTP_404_NOT_FOUND
    except Exception as error:
        logger.error(('Get Store Group raised an exception: ', error.args))
        return {'message': 'maggroups.get_group.error'}, HTTP_500_INTERNAL_SERVER_ERROR


@maggroups_blueprint.route('/storenogroups', methods=['GET'])
@jwt_required()
def get_stores_wo_group():
    query = Magasin.select(
            Magasin.pk, Magasin.mag, Magasin.libelle
        ).where(
            Magasin.groupe == ''
        ).order_by(
            Magasin.mag.asc()
        )

    return {'data': [
        {"pk": store.pk, "mag": store.mag, "libelle": store.libelle} for store in query]}, HTTP_200_OK


@maggroups_blueprint.route('/move/<string:id>', methods=['PATCH'])
@jwt_required()
def move_store(id):
    try:

        data = request.get_json()
        source = data.get('source', '') or ''
        destination = data.get('destination', '')

        if source != '':
            store = Magasin.get(pk=id, groupe=source)
        else:
            store = Magasin.get(pk=id)

        if destination != '':
            MagasinGroupe.get(code=destination)

        previousGroupCode = store.groupe
        store.groupe = destination
        store.save()

        log_mag_groups(session['username'], 'move_store_group',
                       f'Updated store (pk:{store.pk}) groupe from [{previousGroupCode}]  to [{destination}]')

        return {}, HTTP_204_NO_CONTENT

    except Magasin.DoesNotExist:
        logger.error('Move Store : Store with id %s does not exist', id)
        return {'message': 'maggroups.error.store_not_exist'}, HTTP_404_NOT_FOUND
    except MagasinGroupe.DoesNotExist:
        logger.error('Move Store: Group with code %s does not exist', destination)
        return {'message': 'maggroups.error.group_not_exist'}, HTTP_404_NOT_FOUND
    except Exception as ex:
        logger.error(ex)
        return {'message': 'maggroups.move.error'}, HTTP_500_INTERNAL_SERVER_ERROR


def create_group(code, name):
    """Create or update the group for astus"""
    logger.info(f"Create groupe {name} for {code}")
    newGroup, _ = MagasinGroupe.get_or_create(code=code, defaults={
        "nom": name,
    })
    return newGroup


@maggroups_blueprint.route('/add', methods=['PUT'])
@jwt_required()
def add_group():
    try:
        data = request.get_json()

        code = data.get('code', None)
        name = data.get('name', None)
        newGroup = create_group(code, name)

        log_mag_groups(session['username'], 'add_store_group',
                       f'Added new group (pk:{newGroup.pk})')

        return {}, HTTP_204_NO_CONTENT

    except Exception as ex:
        logger.error(ex)
        return {'message': 'maggroups.add.error'}, HTTP_500_INTERNAL_SERVER_ERROR


@maggroups_blueprint.route('/<string:id>', methods=['DELETE'])
@jwt_required()
def delete_group(id):
    try:
        group = MagasinGroupe.get(pk=id)
        stores = Magasin.select(Magasin.pk).where(Magasin.groupe == group.code)

        if stores.count() == 0:
            group.delete_instance()
            log_mag_groups(session['username'], 'delete_group',
                           f'Deleted group (pk:{group.pk})')
            return {}, HTTP_204_NO_CONTENT
        else:
            raise IntegrityError('Cannot delete group with associated stores')

    except DoesNotExist as ex:
        logger.error(ex)
        return {'message': 'maggroups.error.group_not_exist'}, HTTP_404_NOT_FOUND
    except IntegrityError as ex:
        logger.error(ex)
        return {'message': 'maggroups.error.group_has_stores'}, HTTP_409_CONFLICT
    except Exception as ex:
        logger.error(ex)
        return {'message': 'maggroups.delete.error'}, HTTP_500_INTERNAL_SERVER_ERROR


@maggroups_blueprint.route('/update/<string:id>', methods=['PATCH'])
@jwt_required()
def update_group(id):
    try:
        data = request.get_json()
        groupe = MagasinGroupe.get(pk=id)

        if 'name' in data and data['name']:
            groupe.nom = data['name']
            groupe.save()
            log_mag_groups(session['username'], 'update_group_code',
                           f'Updated group (pk:{groupe.pk}) name to [{data["name"]}]')

        if 'code' in data and data['code']:
            with mysql_db.atomic():
                previousCode = groupe.code
                groupe.code = data['code']
                groupe.save()

                query = Magasin.update(groupe=groupe.code).where(Magasin.groupe == previousCode)
                query.execute()

                log_mag_groups(session['username'], 'update_group_code',
                               f'Updated group (pk:{groupe.pk}) code to [{data["code"]}]')

        return {'message': 'maggroups.update.success'}, HTTP_200_OK
    except MagasinGroupe.DoesNotExist:
        return {'message': 'maggroups.error.group_not_exist'}, HTTP_404_NOT_FOUND
    except IntegrityError:
        logger.error('Update group: Duplicate entry for code %s', data['code'])
        return {'message': 'maggroups.update.error.duplicate_entry'}, HTTP_409_CONFLICT
    except Exception as ex:
        logger.error(ex)
        return {'message': 'maggroups.update.error'}, HTTP_500_INTERNAL_SERVER_ERROR


def log_mag_groups(username: str, action: str, message: str):
    """
    Add new log for Magasin Groupe

    :param username: User made the action to log
    :param action:
    :param message: message to log
    """
    logger.info('MagGroups[%s](%s)): %s' % (action, username, message))
    wlog = WebLogActions()
    wlog.chrono = datetime.now()
    wlog.username = username
    wlog.equipement_type = ''
    wlog.action = action
    wlog.message = message
    wlog.save()
